using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;

namespace SPStudio
{
	/// <summary>
	/// Displays and image within a window.
	/// Allows for the Rubberband Line
	/// style of data entry.
	/// </summary>
	public class ImageContainer : System.Windows.Forms.Form
	{
		#region Public Vars
		/// <summary>
		/// Handle to parent workspace.
		/// </summary>
		public Workspace workspace = null;

		/// <summary>
		/// Flag to indicate the success of creation.
		/// </summary>
		public bool creationCanceled = false;
		
		/// <summary>
		/// Flag to indicate that the container
		/// has been removed.
		/// </summary>		
		public bool removed = false;
		
		/// <summary>
		/// Once the 2D data is converted
		/// to 3D data some of the information
		/// is recorded for optimization
		/// purposes within other classes.
		/// </summary>
		public double greatestX = 0.0;
		public double greatestY = 0.0;
		public double smallestY = 0.0;
		public double distance_to_camera = 0.0;
		
		/// <summary>
		/// The location of the image file.
		/// </summary>
		public String imageFile;
		
		/// <summary>
		/// The reference degree the image was
		/// taken from.
		/// </summary>
		public double degree;				
		
		/// <summary>
		/// This ArrayList contains ArrayLists
		/// of 2D Points, called referred to as 
		/// point groups.
		/// </summary>
		public ArrayList groups;		
		#endregion

		#region Private Vars	
		private Bitmap image;

		/// <summary>
		/// Indicates the current
		/// Mode of the ImageContainer.
		/// There are two modes.
		/// A user can Add new points,
		/// or they can Move existing
		/// points.
		/// </summary>
		private bool addMode = true;

		/// <summary>
		/// Add and Edit mode variables.
		/// </summary>
		private Point p0;
		private Point p1;
		private Point p2;
		private bool firstPoint = true;
		private bool firstClick = true;
		private bool trackMouse = false;
		private bool dragPoint = false;
		private int groupIndex = -1;
		private int pointIndex = -1;
		private int mouseAtX = 0;
		private int mouseAtY = 0;
		#endregion

		#region Windows Generated Vars
		private System.Windows.Forms.PictureBox ScratchPad;
		private System.Windows.Forms.MainMenu ImageMenu;
		private System.Windows.Forms.MenuItem EditMenu;
		private System.Windows.Forms.MenuItem RemoveEditItem;
		private System.Windows.Forms.MenuItem DegreeEditItem;
		private System.Windows.Forms.MenuItem ModeMenu;
		private System.Windows.Forms.MenuItem AddModeMenuItem;
		private System.Windows.Forms.MenuItem EditModeMenuItem;

		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;
		#endregion

		#region Constructors
		/// <summary>
		/// Creates an instance of ImageAssociator
		/// in order to gather the needed
		/// initialization information.
		/// Obtains a handle to the parent
		/// workspace.
		/// </summary>
		/// <param name="Parent"></param>
		public ImageContainer(Workspace Parent)
		{
			InitializeComponent();
			
			workspace = Parent;

			//Create an ImageAssociator
			//and block control from the
			//desktop.
			ImageAssociator ia = new ImageAssociator();
			ia.ShowDialog();
			if(ia.canceled)
			{
				ia.Close();
				creationCanceled = true;
			}
			else
			{
				imageFile = ia.imageFile.Trim();
				degree = ia.degree;
				ia.Close();
				try
				{
					image = new Bitmap(imageFile);

					//Obtain the current screen resolution
					Screen Scr = Screen.FromControl(workspace.desktop);
					int ScrHeight = Scr.WorkingArea.Height;
					int ScrWidth = Scr.WorkingArea.Width;

					//If the image is larger than the current
					//screen resolution that cancel the creation.
					//This is due to .NET having issues with properly
					//displaying images within windows that are
					//larger than the current screen.
					if((ScrHeight>=image.Size.Height)&&(ScrWidth>=image.Size.Width))
					{
						//workspace.setImageSize(Size) will set the
						//workspace's size to the image size IF
						//a size has not already been set. Otherwise
						//this method call does nothing.
						workspace.setImageSize(image.Size);

						//Only allow for images to be added
						//if they are the same resolution
						//as all previously added images.
						if(workspace.verifySize(image.Size))
						{
							greatestX = (double)(-1)*image.Size.Width;
							greatestY = (double)(-1)*image.Size.Height;
							smallestY = (double)image.Size.Height;

							ClientSize = image.Size;
							ScratchPad.Image = image;
							Text = degree.ToString() + ":Add Mode";
							groups = new ArrayList(1);
						}
						else
						{
							MessageBox.Show("Image Size is not compatible with workspace");
							creationCanceled = true;
						}
					}
					else
					{
						MessageBox.Show("Image Size is too large for current screen resolution"+
							"\r\nCurrent Screen Resolution: "+ScrWidth.ToString()+"X"+ScrHeight.ToString()+"\r\n"+
							"Image Resolution: "+image.Size.Width.ToString()+"X"+image.Size.Height.ToString());
						creationCanceled = true;
					}
				}
				catch(ArgumentException)
				{
					MessageBox.Show("Invalid Image File: "+imageFile);
					ia.Close();
					creationCanceled = true;
				}
			}
		}

		/// <summary>
		/// Constructor used if initialization
		/// data has already been obtained
		/// through File I/O.
		/// Assumptions are made that the
		/// data doesn't need to be tested for
		/// validity since in order to be saved
		/// the data must have been validated
		/// previously.
		/// </summary>
		/// <param name="Parent"></param>
		/// <param name="ImageFile"></param>
		/// <param name="Degree"></param>
		/// <param name="Groups"></param>
		public ImageContainer(Workspace Parent, String ImageFile, double Degree, ArrayList Groups)
		{
			InitializeComponent();
			
			workspace = Parent;
			imageFile = ImageFile;
			image = new Bitmap(ImageFile);
			degree = Degree;

			greatestX = (double)(-1)*image.Size.Width;
			greatestY = (double)(-1)*image.Size.Height;
			smallestY = (double)image.Size.Height;

			ClientSize = image.Size;
			ScratchPad.Image = image;
			Text = degree.ToString() + ":Add Mode";
			ClientSize = image.Size;
			groups = Groups;
			
			MdiParent = Parent.desktop;
		}
		#endregion

		#region ToString Override
		/// <summary>
		/// Overridden to properly display
		/// the information within
		/// the ImageList.
		/// </summary>
		/// <returns></returns>
		public override string ToString()
		{
			return degree.ToString()+" Degrees : "+imageFile;
		}
		#endregion

		#region Dispose Operation
		/// <summary>
		/// Clean up any resources being used.
		/// Verify that the Form is no longer
		/// being used, if so then dispose,
		/// otherwise just hide it.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if(removed || creationCanceled || workspace.closing)
			{
				if( disposing )
				{
					if(components != null)
					{
						components.Dispose();
					}
				}
				base.Dispose( disposing );
			}
			else
			{
				Hide();
			}
		}
		#endregion

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.ScratchPad = new System.Windows.Forms.PictureBox();
			this.ImageMenu = new System.Windows.Forms.MainMenu();
			this.EditMenu = new System.Windows.Forms.MenuItem();
			this.RemoveEditItem = new System.Windows.Forms.MenuItem();
			this.DegreeEditItem = new System.Windows.Forms.MenuItem();
			this.ModeMenu = new System.Windows.Forms.MenuItem();
			this.AddModeMenuItem = new System.Windows.Forms.MenuItem();
			this.EditModeMenuItem = new System.Windows.Forms.MenuItem();
			this.SuspendLayout();
			// 
			// ScratchPad
			// 
			this.ScratchPad.Dock = System.Windows.Forms.DockStyle.Fill;
			this.ScratchPad.Location = new System.Drawing.Point(0, 0);
			this.ScratchPad.Name = "ScratchPad";
			this.ScratchPad.Size = new System.Drawing.Size(292, 266);
			this.ScratchPad.TabIndex = 0;
			this.ScratchPad.TabStop = false;
			this.ScratchPad.MouseUp += new System.Windows.Forms.MouseEventHandler(this.ScratchPad_MouseUp);
			this.ScratchPad.MouseMove += new System.Windows.Forms.MouseEventHandler(this.ScratchPad_MouseMove);
			this.ScratchPad.MouseDown += new System.Windows.Forms.MouseEventHandler(this.ScratchPad_MouseDown);
			// 
			// ImageMenu
			// 
			this.ImageMenu.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					  this.EditMenu,
																					  this.ModeMenu});
			// 
			// EditMenu
			// 
			this.EditMenu.Index = 0;
			this.EditMenu.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					 this.RemoveEditItem,
																					 this.DegreeEditItem});
			this.EditMenu.Text = "Edit";
			// 
			// RemoveEditItem
			// 
			this.RemoveEditItem.Index = 0;
			this.RemoveEditItem.Text = "Remove";
			this.RemoveEditItem.Click += new System.EventHandler(this.RemoveEditItem_Click);
			// 
			// DegreeEditItem
			// 
			this.DegreeEditItem.Index = 1;
			this.DegreeEditItem.Text = "Re-Associate";
			this.DegreeEditItem.Click += new System.EventHandler(this.DegreeEditItem_Click);
			// 
			// ModeMenu
			// 
			this.ModeMenu.Index = 1;
			this.ModeMenu.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					 this.AddModeMenuItem,
																					 this.EditModeMenuItem});
			this.ModeMenu.Text = "Mode";
			// 
			// AddModeMenuItem
			// 
			this.AddModeMenuItem.Index = 0;
			this.AddModeMenuItem.Text = "Add";
			this.AddModeMenuItem.Click += new System.EventHandler(this.AddModeMenuItem_Click);
			// 
			// EditModeMenuItem
			// 
			this.EditModeMenuItem.Index = 1;
			this.EditModeMenuItem.Text = "Edit";
			this.EditModeMenuItem.Click += new System.EventHandler(this.EditModeMenuItem_Click);
			// 
			// ImageContainer
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.ClientSize = new System.Drawing.Size(292, 266);
			this.Controls.Add(this.ScratchPad);
			this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.Fixed3D;
			this.MaximizeBox = false;
			this.Menu = this.ImageMenu;
			this.Name = "ImageContainer";
			this.Text = "ImageContainer";
			this.KeyDown += new System.Windows.Forms.KeyEventHandler(this.ImageContainer_KeyDown);
			this.LocationChanged += new System.EventHandler(this.ImageContainer_LocationChanged);
			this.Enter += new System.EventHandler(this.ImageContainer_Enter);
			this.ResumeLayout(false);

		}
		#endregion

		#region Remove Image Container Handler
		/// <summary>
		/// Set the removed flag so
		/// proper disposal will be done.
		/// Remove the ImageContainer from
		/// the workspace.
		/// Then close the form.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void RemoveEditItem_Click(object sender, System.EventArgs e)
		{
			removed = true;
			workspace.removeImage(this);
			Close();
		}
		#endregion

		#region ScratchPad Controls
		/// <summary>
		/// Handle Mouse Click events on the ScratchPad.
		/// Based on the current mode and the button
		/// pressed perform different operations.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ScratchPad_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			if(!addMode)
			{
				//Now in EDIT mode
				if(e.Button == MouseButtons.Left)
				{
					//Don't allow a user to
					//switch modes while dragging
					//a point.
					ModeMenu.Enabled = false;

					//Find the Point that the user
					//has clicked on and 
					//take control of it by obtaining
					//its location in the groups ArrayList
					IEnumerator groupWalk = groups.GetEnumerator();
					int tempPointIndex = 0;
					int tempGroupIndex = 0;
					while(groupWalk.MoveNext())
					{
						tempPointIndex = 0;
						IEnumerator pointWalk = ((ArrayList)groupWalk.Current).GetEnumerator();
						while(pointWalk.MoveNext())
						{
							Point temp = (Point)pointWalk.Current;
							if((temp.X+2>e.X)&&(temp.X-2<e.X)&&(temp.Y+2>e.Y)&&(temp.Y-2<e.Y))
							{
								groupIndex = tempGroupIndex;
								pointIndex = tempPointIndex;
								dragPoint = true;
							}
							tempPointIndex++;
						}
						tempGroupIndex++;
					}
				}
			}
			else
			{
				//Now in ADD mode
				if(e.Button == MouseButtons.Left)
				{
					if(firstClick)
					{
						//Don't allow a user to
						//switch modes while
						//a Rubberband Line is
						//active.
						ModeMenu.Enabled = false;

						//This is the first click
						//so create a new point group
						//and the first point.
						//Begin the Rubberband line.
						trackMouse = true;
						firstClick = false;
						p1 = new Point(e.X,e.Y);
						if(firstPoint)
						{
							//Create the new point group
							groups.Add(new ArrayList(1));
							groups.Capacity++;
							firstPoint = false;
							//p0 is the first point in the group.
							//It is remembered so that a 
							//closed polygon can be generated
							//when the right mouse button is
							//clicked.
							p0 = p1;
						}

						//Go to the very last entry in the groups ArrayList
						//and add p1 to it.
						((ArrayList)groups[groups.Count-1]).Add(p1);
						((ArrayList)groups[groups.Count-1]).Capacity++;
					}
					else
					{
						//This is not the first click
						if(!((p1.X==e.X)&&(p1.Y==e.Y)))
						{
							//The user is not trying to put two points in the same
							//location one after the other. This is to handle
							//accidental double clicks.

							//p2 is the new point where the user clicked.
							p2 = new Point(e.X,e.Y);

							//Go to the last point group within the groups ArrayList
							//and add p2.
							((ArrayList)groups[groups.Count-1]).Add(p2);
							((ArrayList)groups[groups.Count-1]).Capacity++;
							
							//Now that p2 is added me reassign p1.
							p1 = p2;
						}
					}
				}
				if(e.Button == MouseButtons.Right)
				{
					//The point group is completed.
					//Reset state in order to 
					//accept a new group of points.
					firstClick = true;
					firstPoint = true;
					trackMouse = false;
					ModeMenu.Enabled = true;

					//If any of the groups contain
					//less than 3 points then
					//they are not polygons, they
					//are lines, so remove them.
					int killIndex = 0;
					IEnumerator groupWalk = groups.GetEnumerator();
					while(groupWalk.MoveNext())
					{
						if(((ArrayList)groupWalk.Current).Count<3)
						{
							groups.RemoveAt(killIndex);
							groupWalk = groups.GetEnumerator();
						}
						killIndex++;
					}
				}
			}
			Refresh();
		}

		/// <summary>
		/// Handle the Rubberband line effect for
		/// the data entry.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ScratchPad_MouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			mouseAtX = e.X;
			mouseAtY = e.Y;
			
			//Prevent the line from accepting
			//movement changes outside of the image.
			if(mouseAtX<0) mouseAtX=0;
			if(mouseAtX>ClientSize.Width) mouseAtX = ClientSize.Width;
			if(mouseAtY<0) mouseAtY=0;
			if(mouseAtY>ClientSize.Height) mouseAtY = ClientSize.Height;
			if(!addMode)
			{
				//In EDIT mode.
				if(dragPoint)
				{
					//Move the selected point to the current mouse location
					((ArrayList)groups[groupIndex])[pointIndex] = new Point(mouseAtX,mouseAtY);
					Refresh();
				}
			}
			else
			{
				//In Add mode
				if(trackMouse)
				{
					//If a Rubberband Line is
					//active then simply refresh.
					Refresh();
				}
			}
		}

		/// <summary>
		/// Handles the releasing of
		/// a point in EDIT mode.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ScratchPad_MouseUp(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			if(!addMode)
			{
				//In EDIT mode

				//Reset the Edit mode variables
				//to accept a new point to drag
				dragPoint = false;
				groupIndex = -1;
				pointIndex = -1;
				ModeMenu.Enabled = true;
			}
		}
		#endregion

		#region Code To Handle Keyboard Input
		/// <summary>
		/// Pressing the ESC key will back up one point at a time
		/// when in ADD mode, and will remove a selected point when
		/// in EDIT mode.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ImageContainer_KeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			if(e.KeyCode == Keys.Escape)
			{
				if(addMode)
				{
					//In ADD mode
					if(trackMouse)
					{
						//Rubberband Line is active.
						
						//Remove the last point entered
						((ArrayList)groups[groups.Count-1]).RemoveAt(((ArrayList)groups[groups.Count-1]).Count-1);
						((ArrayList)groups[groups.Count-1]).Capacity--;
							
						//If the user has removed points
						//such that less than 3 remain in
						//the group then cancel the
						//creation all together.
						int killIndex = 0;
						IEnumerator groupWalker = groups.GetEnumerator();
						while(groupWalker.MoveNext())
						{
							if(((ArrayList)groupWalker.Current).Count < 3)
							{								
								ModeMenu.Enabled = true;
								firstClick = true;
								firstPoint = true;
								trackMouse = false;
								groups.RemoveAt(killIndex);
								groupWalker = groups.GetEnumerator();
							}
							killIndex++;
						}
						Refresh();

						if(groups.Count>0)
						{
							//Assign p1 to be the last point of the last group
							ArrayList lastPointGroup = (ArrayList)groups[groups.Count-1];
							p1 = (Point)lastPointGroup[lastPointGroup.Count-1];
						}
					}
				}
				else
				{
					//In EDIT mode
					if((groupIndex>(-1))&&(pointIndex>(-1)))
					{
						//Remove the selected point
						((ArrayList)groups[groupIndex]).RemoveAt(pointIndex);
						
						//Reset the Edit mode variables
						//to accept a new point to drag
						ModeMenu.Enabled = true;
						dragPoint = false;
						groupIndex = -1;
						pointIndex = -1;

						//If the user has deleted points
						//in such a way that a group now
						//has less than 3 points, remove
						//the group.
						int killIndex = 0;
						IEnumerator groupWalker = groups.GetEnumerator();
						while(groupWalker.MoveNext())
						{
							if(((ArrayList)groupWalker.Current).Count < 3)
							{
								groups.RemoveAt(killIndex);
								groupWalker = groups.GetEnumerator();
							}
							killIndex++;
						}
						Refresh();
					}
				}				
			}		
		}
		#endregion

		#region Refresh Override
		/// <summary>
		/// Overridden Refresh uses the groups of
		/// points to display normal lines, as well
		/// as the Rubberband Line when active.
		/// </summary>
		public override void Refresh()
		{
			base.Refresh ();
			IEnumerator groupWalk = groups.GetEnumerator();
			Graphics g = ScratchPad.CreateGraphics();
			Pen pen = new Pen(Color.Turquoise);
			Pen pointPen = new Pen(Color.Red);
			pen.Width = 2;
			pointPen.Width = 4;
			pen.DashStyle = DashStyle.Solid;
			pointPen.DashStyle = DashStyle.Solid;

			Point head = new Point();
			Point first = new Point();
			Point tail = new Point();
			
			int counter = 0;

			while(groupWalk.MoveNext())
			{
				++counter;

				IEnumerator pointWalk = ((ArrayList)groupWalk.Current).GetEnumerator();
				if(pointWalk.MoveNext())
				{
					//Grab information about the first point in the group
					//and display it
					head = (Point)pointWalk.Current;
					first = head;
					tail = first;
					g.DrawEllipse(pointPen, head.X, head.Y, 1, 1);
				}
				while(pointWalk.MoveNext())
				{
					tail = (Point)pointWalk.Current;
					g.DrawLine(pen, head.X, head.Y, tail.X, tail.Y);
					g.DrawEllipse(pointPen, tail.X, tail.Y, 1, 1);
					head = tail;
				}
				if(counter!=groups.Count)
				{
					//We are at the last point and
					//there are still groups to draw.
					g.DrawLine(pen, tail.X, tail.Y, first.X, first.Y);
				}
				else if(!trackMouse)
				{
					//We are at the last point in the last group
					//and the Rubberband Line isn't active.
					g.DrawLine(pen, tail.X, tail.Y, first.X, first.Y);
				}
				else
				{
					//We are at the last point in the last group
					//and the Rubberband Line is active.
					Pen dPen = new Pen(Color.Fuchsia);
					dPen.Width = 2;
					dPen.DashStyle = DashStyle.Dot;
					g.DrawLine(dPen, tail.X, tail.Y, mouseAtX, mouseAtY);
				}
			}
		}

		/// <summary>
		/// Trigger a Refresh.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ImageContainer_LocationChanged(object sender, System.EventArgs e)
		{
			Refresh();
		}

		/// <summary>
		/// Trigger a Refresh.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ImageContainer_Enter(object sender, System.EventArgs e)
		{
			Refresh();		
		}
		#endregion

		#region Mode Menu Item Handlers
		/// <summary>
		/// Switch the data input method
		/// to ADD mode.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void AddModeMenuItem_Click(object sender, System.EventArgs e)
		{
			addMode = true;
			Text = degree.ToString() +":Add Mode";
		}

		/// <summary>
		/// Switch the data input method
		/// to EDIT mode.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void EditModeMenuItem_Click(object sender, System.EventArgs e)
		{
			addMode = false;
			Text = degree.ToString() +":Edit Mode";
		}
		#endregion

		#region Prepare and Retrieve Data for Algorithm
		/// <summary>
		/// Calls private methods sequentially in order
		/// to return an Array List of rotated Triangle3Ds
		/// to be used by the VRMC algorithm.
		/// </summary>
		/// <returns></returns>
		public ArrayList getTriangles()
		{
			return rotate3DTriangles(
				convert3DLinesTo3DTriangles(
				convert2DTo3DLines(
				convertPointGroupsTo2DLines())));
		}

		/// <summary>
		/// Converts the points within the groups
		/// ArrayList to Line2Ds
		/// </summary>
		/// <returns></returns>
		private ArrayList convertPointGroupsTo2DLines()
		{
			ArrayList line2DArray = new ArrayList(1);
			Point p1;
			Point p2;

			//Traverse the groups ArrayList
			for(int groupIndex=0; groupIndex<groups.Count; groupIndex++)
			{
				ArrayList points = (ArrayList)groups[groupIndex];
				//Traverse the points ArrayList
				//creating and adding lines as we traverse.
				for(int pointIndex=1; pointIndex<points.Count; pointIndex++)
				{
					p1 = (Point)points[pointIndex-1];
					p2 = (Point)points[pointIndex];

					Line2D line = new Line2D(p1,p2);
					line2DArray.Add(line);
					line2DArray.Capacity++;
				}
				//Connect the first and last points in the group
				//to create the final line.
				line2DArray.Add(new Line2D((Point)points[points.Count-1],(Point)points[0]));
				line2DArray.Capacity++;
			}
			return line2DArray;
		}

		/// <summary>
		/// Convert the Line2Ds to Line3Ds by
		/// calling the convertAndTranslate method
		/// contained within each Line2D.
		/// </summary>
		/// <param name="Line2DArray"></param>
		/// <returns></returns>
		private ArrayList convert2DTo3DLines(ArrayList Line2DArray)
		{
			ArrayList line3DArray = new ArrayList(1);

			//Traverse the ArrayList of Line2Ds
			for(int lineIndex=0; lineIndex < Line2DArray.Count; lineIndex++)
			{
				//Line2D has a public method to convert
				//itself into a Line3D given the image size
				Line3D line = ((Line2D)Line2DArray[lineIndex]).convertAndTranslate(image.Size);

				//Record the greatest X value encountered
				if(Math.Abs(line.p1.x)>greatestX)
				{
					greatestX = Math.Abs(line.p1.x);					
				}
				if(Math.Abs(line.p2.x)>greatestX)
				{
					greatestX = Math.Abs(line.p2.x);
				}

				//Record the greatest and smallest
				//Y values encountered.
				if(line.p1.y>greatestY)
					greatestY = line.p1.y;
				if(line.p1.y<smallestY)
					smallestY = line.p1.y;
				if(line.p2.y>greatestY)
					greatestY = line.p2.y;
				if(line.p2.y<smallestY)
					smallestY = line.p2.y;

				line3DArray.Add(line);
				line3DArray.Capacity++;
			}
			return line3DArray;
		}

		/// <summary>
		/// Convert the Line3Ds into Triangle3Ds
		/// using standard geometric properties.
		/// </summary>
		/// <param name="Line3DArray"></param>
		/// <returns></returns>
		private ArrayList convert3DLinesTo3DTriangles(ArrayList Line3DArray)
		{
			ArrayList triangle3DArray = new ArrayList(1);

			//In order to find the distance to the camera
			//we use the idea that when bisecting the base(image plane)
			//of an isoceles triangle and drawing a line
			//from that point to the tip(camera) we bisect
			//the FOV of the camera. Using this angle, and the
			//distance of half of the longest dimension of the image
			//we can calculate the lengh of the line connecting
			//the bisection point to the point where the
			//camea is.
			double tangent_of_fov = Math.Tan((workspace.cameraFOV/2.0)*(Math.PI/180.0));
			double half_long_view;
			half_long_view = (double)(image.Size.Width/2);
			if(half_long_view < (double)(image.Size.Height/2)) 
				half_long_view = (double)(image.Size.Height/2);

			distance_to_camera = half_long_view/tangent_of_fov;
			
			Vector3D camera_at = new Vector3D(0.0,0.0,distance_to_camera);

			//Traverse the Line3D ArrayList and generate the Triangle3Ds
			for(int lineIndex=0; lineIndex<Line3DArray.Count; lineIndex++)
			{
				Line3D line = (Line3D)Line3DArray[lineIndex];
				Vector3D v1 = new Vector3D(line.p1.x*2.0,line.p1.y*2.0,(-1.0)*distance_to_camera);
				Vector3D v2 = new Vector3D(line.p2.x*2.0,line.p2.y*2.0,(-1.0)*distance_to_camera);
				triangle3DArray.Add(new Triangle3D(camera_at,v1,v2));
				triangle3DArray.Capacity++;
			}
			return triangle3DArray;
		}

		/// <summary>
		/// Rotate the Triangle3Ds by rotating each point
		/// in each Triangle3D by the degree specified
		/// within this ImageContainer.
		/// </summary>
		/// <param name="Triangle3DArray"></param>
		/// <returns></returns>
		private ArrayList rotate3DTriangles(ArrayList Triangle3DArray)
		{
			ArrayList rotatedTriangleArray = new ArrayList(1);

			double oldX;
			double oldZ;

			IEnumerator triangleWalk = Triangle3DArray.GetEnumerator();

			//Traverse the ArrayList of Triangle3Ds and rotate them.
			while(triangleWalk.MoveNext())
			{
				Triangle3D t = (Triangle3D)triangleWalk.Current;
				Vector3D v1 = t.p1;
				Vector3D v2 = t.p2;
				Vector3D v3 = t.p3;

				//The following are standard formulas for
				//the rotation of a point around the z axis
				oldX = v1.x;
				oldZ = v1.z;

				v1.x = oldX*Math.Cos(degree*(Math.PI/180))+oldZ*Math.Sin(degree*(Math.PI/180));
				v1.z = (-1.0)*oldX*Math.Sin(degree*(Math.PI/180))+oldZ*Math.Cos(degree*(Math.PI/180));

				oldX = v2.x;
				oldZ = v2.z;

				v2.x = oldX*Math.Cos(degree*(Math.PI/180))+oldZ*Math.Sin(degree*(Math.PI/180));
				v2.z = (-1.0)*oldX*Math.Sin(degree*(Math.PI/180))+oldZ*Math.Cos(degree*(Math.PI/180));

				oldX = v3.x;
				oldZ = v3.z;

				v3.x = oldX*Math.Cos(degree*(Math.PI/180))+oldZ*Math.Sin(degree*(Math.PI/180));
				v3.z = (-1.0)*oldX*Math.Sin(degree*(Math.PI/180))+oldZ*Math.Cos(degree*(Math.PI/180));

				rotatedTriangleArray.Add(new Triangle3D(v1,v2,v3));
				rotatedTriangleArray.Capacity++;
			}
			return rotatedTriangleArray;
		}
		#endregion

		#region Re-Associate Image and/or Degree

		/// <summary>
		/// Allows for the ImageContainer's image and
		/// degree to be changed.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void DegreeEditItem_Click(object sender, System.EventArgs e)
		{
			//Call to special ImageAssociator constructor
			//that pre-sets the degree and image fields.
			//Block control from the desktop.
			ImageAssociator ia = new ImageAssociator(imageFile, degree);
			ia.ShowDialog();
			if(ia.canceled)
			{
				ia.Close();
			}
			else
			{
				imageFile = ia.imageFile.Trim();
				degree = ia.degree;
				ia.Close();
				try
				{
					image = new Bitmap(imageFile);

					//Obtain the current screen resolution
					Screen Scr = Screen.FromControl(workspace.desktop);
					int ScrHeight = Scr.WorkingArea.Height;
					int ScrWidth = Scr.WorkingArea.Width;

					//If the image is larger than the current
					//screen resolution that cancel the creation.
					//This is due to .NET having issues with properly
					//displaying images within windows that are
					//larger than the current screen.
					if((ScrHeight>=image.Size.Height)&&(ScrWidth>=image.Size.Width))
					{
						//workspace.setImageSize(Size) will set the
						//workspace's size to the image size IF
						//a size has not already been set. Otherwise
						//this method call does nothing.
						workspace.setImageSize(image.Size);

						//Only allow for images to be added
						//if they are the same resolution
						//as all previously added images.
						if(workspace.verifySize(image.Size))
						{
							greatestX = (double)(-1)*image.Size.Width;
							greatestY = (double)(-1)*image.Size.Height;
							smallestY = (double)image.Size.Height;

							ClientSize = image.Size;
							ScratchPad.Image = image;
							Text = degree.ToString() + ":Add Mode";
							groups = new ArrayList(1);
						}
						else
						{
							MessageBox.Show("Image Size is not compatible with workspace");
							creationCanceled = true;
						}
					}
					else
					{
						MessageBox.Show("Image Size is too large for current screen resolution"+
							"\r\nCurrent Screen Resolution: "+ScrWidth.ToString()+"X"+ScrHeight.ToString()+"\r\n"+
							"Image Resolution: "+image.Size.Width.ToString()+"X"+image.Size.Height.ToString());
						creationCanceled = true;
					}
				}
				catch(ArgumentException)
				{
					MessageBox.Show("Invalid Image File: "+imageFile);
					ia.Close();
					creationCanceled = true;
					Close();
				}
			}		
		}
		#endregion
	}
}
